/*
	mxMatrix.h version 1.0

	(c) 2002, Joel Hasbrouck, All rights reserved.

	Interface to matlab mx routines.

	To use these classes, just include mxMatrix.h in your mex program:
		#include "mxMatrix.h"

	This code defines matrix classes that implement indexing and bounds checking.
	The indexing functionality allows us to write in a C matlab routine expressions
	of the form "x(i,j)" for the ith row and jth column of a matrix. This avoids the
	necessity of awkward calls to matlab's mxCalcSingleIndex routine.
	The bounds checking functionality facilitates debugging. 
	In production runs, it should be set to zero for efficiency.

	These classes are relatively "light weight". For a full-blown matrix class that
	basically replicates Matlab's functionality in C++, see the mwArray class in
	the Matlab C/C++ Library Toolbox.

	The three user classes are:

	MatrixRhs	A matrix that will be passed back to the
				calling routine (using the prhs[] array).

	MatrixLhs	A matrix that is passed as an argument into the
				calling routine (using the plhs[] array)

	MatrixWork	A matrix that is used within the routine as a computational
				convenience.

	All of these are derived from:

	MatrixBase	A matrix base class that implements indexing and bounds checking.
				The three user classes are derived from MatrixBase.

	The reason for the multiple matrix classes is that in a mex routine, Matlab
	is very picky about when storage is allocated and freed. 
	(1) For a matrix that is passed	as a rhs argument, the space required to hold 
	the data is already allocated by the calling routine and it can only be deallocated 
	by the calling routine. 
	(2) For a matrix that is to be returned as a lhs argument, the space is allocated 
	in the mex routine.
	(3) A matrix that is to be used as a temporary work matrix within the mex routine is
	allocated and deallocated within the routine.

	All of the necessary allocation and deallocation is done by class constructor 
	and destructor funtions. The user should never have to call the mex allocation
	and deallocation routines ("mxCreateDoubleMatrix" and "mxDestroyArray") explicitly,
	when using these classes.

	At the end of this file is a routine:
		double GetDoubleFromStruct(const mxArray *mx, const char *name)
	This routine assumes that mx points to a structure, and "name" is
	a named variable in that structure.  Typically used for extracting
	scalars from an input structure.

*/

#ifndef mxMatrix_H
#define mxMatrix_H

#include <mex.h>
#include <matrix.h>

#ifndef BOUNDS_CHECK
#define BOUNDS_CHECK 0
#endif

/*	In your mex routine, you can turn bounds checking on by setting this variable to
	"1" before including the mxMatrix header file:

	#define BOUNDS_CHECK 1
	#include "mxMatrix.h"

*/

class MatrixBase
{
public:
	double *Data;	//	Pointer to first element of matrix.
	const int *n;	//	Array of dimensions.
	int length;		//	Number of elements.

	//	Utility routines to compute indexes
	int Index(int k1) const;
	int Index(int k1, int k2) const;

	//	Element access
	double & operator () (int k1) { return Data[Index(k1)]; }
	double & operator ()(int k1, int k2) { return Data[Index(k1,k2)]; }

	void Initialize(const mxArray *);
	void Normalize();
	void Print(const char*);

	inline int nRows() {return n[0];}
	const inline int nCols() {return n[1];}

};
void MatrixBase::Initialize(const mxArray *mx)
{
	if (!mx) mexErrMsgTxt("MatrixBase. mx argument is null.\n");
	if (!mxIsDouble(mx)) mexErrMsgTxt("MatrixBase. Argument is not a double.\n");
	int nDim = mxGetNumberOfDimensions(mx);
	if (nDim!=2) mexErrMsgTxt("MatrixBase. Argument is not 2-dimensional.\n");
	n = mxGetDimensions(mx);
	length = mxGetNumberOfElements(mx);
	Data = mxGetPr(mx);
}
//	Indexing & element access
#if BOUNDS_CHECK
int MatrixBase::Index(int k1) const {
	if (k1>length) mexErrMsgTxt("MatrixBase. Index out of bounds.\n");
	return k1-1;
};
int MatrixBase::Index(int k1, int k2) const {
	if (k1>n[0]) mexErrMsgTxt("MatrixBase. Row index out of range.\n");
	if (k2>n[1]) mexErrMsgTxt("MatrixBase. Column index out of range.\n");
	return n[0]*(k2-1)+k1-1;
};
#else
inline int MatrixBase::Index(int k1) const { return k1-1; };
inline int MatrixBase::Index(int k1, int k2) const { return n[0]*(k2-1)+k1-1; };
#endif

void MatrixBase::Normalize()
//	Normalize sum of matrix elements to unity. (Mostly used for probabilities.)
{
	double s=0;
	for (int i=0; i<length; i++) s += Data[i];
	for (	 i=0; i<length; i++) Data[i] = Data[i]/s;
}

void MatrixBase::Print(const char* name)
{
	mexPrintf("%s:\n",name);
	for (int i=1; i<=n[0]; i++)
	{
		for (int j=1; j<=n[1]; j++) mexPrintf("%12.5g",Data[Index(i,j)]);
		mexPrintf("\n");
	}
}

//----------------------------------------------------------------------------------------

class MatrixLhs : public MatrixBase
{
public:
	mxArray *mx;	//	All matrices are stored as matlab "mx" structures.

	//	Constructor
	//MatrixLhs() {};	//	Default construtor does NOT allocate space.
	MatrixLhs(int k1, int k2);	// This constructor DOES allocate space.

};
MatrixLhs::MatrixLhs(int k1, int k2)
{
	mx = mxCreateDoubleMatrix(k1,k2,mxREAL);
	if (!mx) mexErrMsgTxt("MatrixLhs. Can''t create matrix.\n");
	Initialize(mx);
};




//----------------------------------------------------------------------------------------

class MatrixRhs : public MatrixBase
{
public:
	const mxArray *mx;

	MatrixRhs() {};
	MatrixRhs(const mxArray *mx2);
	MatrixRhs(const mxArray *mx2, const char *name);
	Get(const mxArray *mx2, const char *name);
};
MatrixRhs::MatrixRhs(const mxArray *mx2)
{
	mx = mx2;
	Initialize(mx);
};
MatrixRhs::MatrixRhs(const mxArray *mx2, const char *name)
{
	if (!mxIsStruct(mx2)) mexErrMsgTxt("MatrixRhs. Expecting a structure.\n");
	mxArray *mxa=mxGetField(mx2,0,name);
	if (!mxa) {
		char msg[120];
		sprintf(msg,"MatrixRhs. %s not found in input structure.\n",name);
		mexErrMsgTxt(msg);
	}
	mx = mxa;
	Initialize(mx);
}
MatrixRhs::Get(const mxArray *mx2, const char *name)
{
	if (!mxIsStruct(mx2)) mexErrMsgTxt("MatrixRhs. Expecting a structure.\n");
	mxArray *mxa=mxGetField(mx2,0,name);
	if (!mxa) {
		char msg[120];
		sprintf(msg,"MatrixRhs. %s not found in input structure.\n",name);
		mexErrMsgTxt(msg);
	}
	mx = mxa;
	Initialize(mx);
}

//----------------------------------------------------------------------------------------

class MatrixWork : public MatrixBase
{
public:
	mxArray *mx;
	MatrixWork(int k1, int k2=1);
	MatrixWork(const MatrixWork &);
	~MatrixWork() {mxDestroyArray(mx);}
};
MatrixWork::MatrixWork(int k1, int k2)
{
	mx = mxCreateDoubleMatrix(k1,k2,mxREAL);
	if (!mx) mexErrMsgTxt("MatrixLhs. Can''t create matrix.\n");
	Initialize(mx);
};
MatrixWork::MatrixWork(const MatrixWork &mw)
{
	mx = mxDuplicateArray(mw.mx);
	Initialize(mx);
}


//----------------------------------------------------------------------------------------

double GetDoubleFromStruct(const mxArray *mx, const char *name)
{
	MatrixRhs x(mx,name);
	return x(1);
}


#endif
